/**
 * \file: gst_drm_audio_dec.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * Decoder for Digital Radio Mondiale (DRM) audio superframes
 *
 * \component: multimedia/gst
 *
 * \author: Bodo Winter ADITG/SW1 bwinter@de.adit-jv.com
 *
 * \copyright: (c) 2003 - 2015 ADIT Corporation
 *
 ***********************************************************************/
#include "gst_drm_audio_dec.h"

#include "coresup.h"

#include <gst/base/gstdataqueue.h>

#include <string.h>

#ifndef PACKAGE
#define PACKAGE "gst_drm_audio_dec"
#endif

GST_DEBUG_CATEGORY ( drm_audio_debug);
#define GST_CAT_DEFAULT drm_audio_debug

#define GST_DRM_DEC_DESCRIPTION "Decoder for Digital Radio Mondiale (DRM) audio superframes"

#define DRM_AUDIO_MIME_TYPE "audio/x-drm"

#define AUDIO_CONFIG_ID "codec_data"
#define DRM_PLUS_ID "drmplus"
#define HIGHER_PROTECTED_ID "higher-protected"
#define FRAME_CORRUPT_ID "frame-corrupt"
#define EXT_FRAME_COUNT_ID "ext-frame-count"

/* Fallback buffer sizes (stolen from Fraunhofer example code */
#define IN_BUF_SIZE  ( 8196 )
#define PCM_BUF_SIZE_MAX_FRAME_SIZE    (4096)         /*!< Maximum output frame size (HE-AAC: 1920, xHE-AAC: 4096) */
#define PCM_BUF_SIZE_MAX_N_CH          (2)            /*!< Maximum number of output channels (currently only stereo supported) */
#define PCM_BUF_SIZE_MAX_RESAMP_RATIO  (5)            /*!< Maximum resampling ratio (HE-AAC: 48kHz/12kHz, xHE-AAC: 48kHz/9.6kHz),
                                                           Only applies if resampler is being used. */
#define PCM_BUF_SIZE ( PCM_BUF_SIZE_MAX_FRAME_SIZE * PCM_BUF_SIZE_MAX_N_CH * PCM_BUF_SIZE_MAX_RESAMP_RATIO )

/* DRM specification */
#define CONFIG_AAC_CODING 0
#define CONFIG_XHEAAC_CODING 3

#define CONFIG_AUDIO_CODING_BITMASK 0xC
#define CONFIG_SBR_BITMASK 0x2
#define CONFIG_SAMPLE_RATE_BITMASK 0x70
#define CONFING_AUDIO_MODE_BITMASK 0x80

/* PRQA: Lint Message 826, 160: deactivation because casting mechanism of GObject throws the finding */
/*lint -e826 -e160*/

/* Decoder configuration */
enum
{
  PROP_CONCEAL_FADEOUT_SLOPE = 1,
  PROP_CONCEAL_FADEIN_SLOPE,
  PROP_CONCEAL_MUTE_RELEASE,
  PROP_CONCEAL_COMFORT_NOISE_LEVEL,
  PROP_ACTIVATE_ULP,
  PROP_CURRENT_AU_SIZE,
  PROP_CURRENT_BITRATE,
  PROP_SUPERFRAME_COUNT,
  PROP_ERRORFREE_AU_COUNT,
  PROP_ERRONEOUS_AU_COUNT,
  PROP_SUPERFRAME_PARSE_ERROR_COUNT,
  PROP_LAST_AU_OK,
  PROP_EXTERNAL_FRAME_COUNT,
  PROP_EXTERNAL_FRAME_CORRUPT,
  N_PROPERTIES
};

#define PROP_CONCEAL_FADEOUT_SLOPE_ID "conceal-fadeout-slope"
#define PROP_CONCEAL_FADEIN_SLOPE_ID "conceal-fadein-slope"
#define PROP_CONCEAL_MUTE_RELEASE_ID "conceal-mute-release"
#define PROP_CONCEAL_COMFORT_NOISE_LEVEL_ID "conceal-comfort-noise-level"
#define PROP_ACTIVATE_ULP_ID "activate-ulp"

#define PROP_CURRENT_AU_SIZE_ID "current-au-size"
#define PROP_CURRENT_BITRARE_ID "current-bitrate"
#define PROP_SUPERFRAME_COUNT_ID "superframe-count"
#define PROP_ERRORFREE_AU_COUNT_ID "errorfree-au-count"
#define PROP_ERRONEOUS_AU_COUNT_ID "erroneous-au-count"
#define PROP_SUPERFRAME_PARSE_ERROR_COUNT_ID "superframe-parse-error-count"
#define PROP_LAST_AU_OK_ID "last-au-ok"
#define PROP_EXTERNAL_FRAME_COUNT_ID "external-frame-count"
#define PROP_EXTERNAL_FRAME_CORRUPT_ID "external-frame-corrupt"

#define DEFAULT_CONCEAL_FADEOUT_SLOPE 5
#define DEFAULT_CONCEAL_FADEIN_SLOPE 5
#define DEFAULT_CONCEAL_MUTE_RELEASE 0
#define DEFAULT_CONCEAL_COMFORT_NOISE_LEVEL 127
#define DEFAULT_ACTIVATE_ULP FALSE

#define GST_DRM_DEC_SINK_CAPS \
  GST_STATIC_CAPS (DRM_AUDIO_MIME_TYPE)

#define GST_DRM_DEC_SRC_CAPS \
    GST_STATIC_CAPS ("audio/x-raw-int, " \
        "rate = (int) { 9600, 12000, 16000, 19200, 24000, 32000, 38400, 48000 }, " \
        "channels = (int) [ 1, 2 ], " \
        "endianness = (int) BYTE_ORDER, " \
        "width = (int) 16")

static GstStaticPadTemplate sink_factory = GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK, GST_PAD_ALWAYS, GST_DRM_DEC_SINK_CAPS);

static GstStaticPadTemplate src_factory = GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC, GST_PAD_ALWAYS, GST_DRM_DEC_SRC_CAPS);

/* PRQA: Lint Message 19, 123, 144, 751: deactivation because macro is related to GStreamer framework */
/*lint -e19 -e123 -e144 -e751 */
GST_BOILERPLATE (GstDrmDec, drm_dec, GstElement, GST_TYPE_ELEMENT);
/*lint +e19 +e123 +e144 +e751 */

static GParamSpec *obj_properties[N_PROPERTIES] = { NULL, };


/* General functions */
static void drm_dec_finalize (GObject * object);
static void drm_dec_reset (GstDrmDec * dec);
static void drm_dec_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void drm_dec_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

/* State transition functions */
static GstStateChangeReturn drm_dec_change_state (GstElement * element,
    GstStateChange transition);
static gboolean drm_dec_start (GstDrmDec * dec);
static gboolean drm_dec_stop (GstDrmDec * dec);
static void drm_dec_get_buffer_sizes (GstDrmDec * dec, guint *inbuffersize,
    guint *outbuffersize);

static gboolean drm_dec_event (GstPad * pad, GstEvent * event);

/* Configuration functions */
static gboolean drm_dec_config_dec (GstDrmDec * dec, guchar* config, guint configSize,
    guint superFrameBytes, guint lengthHigherProtected, gboolean drmPlus);

/* Processing functions */
static GstFlowReturn drm_dec_chain (GstPad * pad, GstBuffer * buffer);
static GstFlowReturn drm_dec_handle_superframe (GstDrmDec * dec, GstBuffer * buffer);
static int drm_dec_decode_audio_frame (GstDrmDec * dec, guchar * inBuffer, guint* bytesValid,
    INT_PCM * outBuffer);
static void drm_dec_push_to_out_pad (GstDrmDec * dec, INT_PCM * outData,
    guint outNumberSamples);

/**
 * Initialize class properties
 */
static void
drm_dec_base_init (gpointer g_class)
{
  GST_DEBUG_OBJECT (g_class, "drm_dec_base_init");
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_add_static_pad_template (element_class, &src_factory);
  gst_element_class_add_static_pad_template (element_class, &sink_factory);

  gst_element_class_set_details_simple (element_class, "DRM audio decoder",
      "Codec/Decoder/Audio", GST_DRM_DEC_DESCRIPTION,
      "Bodo Winter <bwinter@de.adit-jv.com>");
}

/**
 * One-time initializer for virtual functions and properties
 */
static void
drm_dec_class_init (GstDrmDecClass * klass)
{
  GST_DEBUG_OBJECT (klass, "drm_dec_class_init");
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  GstElementClass *base_class = GST_ELEMENT_CLASS (klass);

  gobject_class->set_property = GST_DEBUG_FUNCPTR (drm_dec_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (drm_dec_get_property);
  gobject_class->finalize = GST_DEBUG_FUNCPTR (drm_dec_finalize);

  base_class->change_state = GST_DEBUG_FUNCPTR (drm_dec_change_state);

  obj_properties[PROP_CONCEAL_FADEOUT_SLOPE] =
      g_param_spec_uint (PROP_CONCEAL_FADEOUT_SLOPE_ID, PROP_CONCEAL_FADEOUT_SLOPE_ID,
          "AAC Error concealment: Length of the audio signals fade-out slope " \
          "(in frames) applied for multiple consecutive corrupted audio frames.",
          0, 16, DEFAULT_CONCEAL_FADEOUT_SLOPE,
          G_PARAM_READABLE |G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_STRINGS);

  obj_properties[PROP_CONCEAL_FADEIN_SLOPE] =
      g_param_spec_uint (PROP_CONCEAL_FADEIN_SLOPE_ID, PROP_CONCEAL_FADEIN_SLOPE_ID,
          "AAC Error concealment: Analogous to the conceal-fadeout-slope parameter " \
          "but for the fade-in slope applied to the audio output signal after " \
          "error-free frames have arrived.",
          0, 16, DEFAULT_CONCEAL_FADEIN_SLOPE,
          G_PARAM_READABLE |G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_STRINGS);

  obj_properties[PROP_CONCEAL_MUTE_RELEASE] =
      g_param_spec_uint (PROP_CONCEAL_MUTE_RELEASE_ID, PROP_CONCEAL_MUTE_RELEASE_ID,
          "AAC Error concealment: The number of error-free frames which have to " \
          "arrive at the decoder (while muting) until the signal gets fade-in " \
          "again. This parameter can be used to avoid a pumping output signal " \
          "that can be generated by consecutive sequences of fade-in and out.",
          0, 16, DEFAULT_CONCEAL_MUTE_RELEASE,
          G_PARAM_READABLE |G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_STRINGS);

  obj_properties[PROP_CONCEAL_COMFORT_NOISE_LEVEL] =
        g_param_spec_uint (PROP_CONCEAL_COMFORT_NOISE_LEVEL_ID, PROP_CONCEAL_COMFORT_NOISE_LEVEL_ID,
            "AAC Error concealment: The level of the noise signal that " \
            "will be inserted after a fade-out due to corrupt input data: " \
            "0 (full scaled noise) to 127 (no noise insertion).",
            0, 127, DEFAULT_CONCEAL_COMFORT_NOISE_LEVEL,
            G_PARAM_READABLE |G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_STRINGS);

  obj_properties[PROP_ACTIVATE_ULP] =
      g_param_spec_boolean (PROP_ACTIVATE_ULP_ID, PROP_ACTIVATE_ULP_ID,
          "Replaces the USAC decoder part with a special non-standard conform " \
          "low power implementation",
          DEFAULT_ACTIVATE_ULP,
          G_PARAM_READABLE |G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_STRINGS);

  obj_properties[PROP_CURRENT_AU_SIZE] =
      g_param_spec_uint (PROP_CURRENT_AU_SIZE_ID, PROP_CURRENT_AU_SIZE_ID,
          "Transformation length of the used codec per frame (Samples)", 0,
          G_MAXUINT, 0, G_PARAM_READABLE | G_PARAM_STATIC_STRINGS);

  obj_properties[PROP_CURRENT_BITRATE] =
      g_param_spec_uint (PROP_CURRENT_BITRARE_ID, PROP_CURRENT_BITRARE_ID,
          "The audio bitrate (bit/s)", 0, G_MAXUINT, 0,
          G_PARAM_READABLE | G_PARAM_STATIC_STRINGS);

  obj_properties[PROP_SUPERFRAME_COUNT] =
      g_param_spec_uint (PROP_SUPERFRAME_COUNT_ID, PROP_SUPERFRAME_COUNT_ID,
          "Number of decoded super frames", 0, G_MAXUINT, 0,
          G_PARAM_READABLE | G_PARAM_STATIC_STRINGS);

  obj_properties[PROP_ERRORFREE_AU_COUNT] =
      g_param_spec_uint (PROP_ERRORFREE_AU_COUNT_ID, PROP_ERRORFREE_AU_COUNT_ID,
          "Number of audio frames which have been decoded successfully", 0,
          G_MAXUINT, 0, G_PARAM_READABLE | G_PARAM_STATIC_STRINGS);

  obj_properties[PROP_ERRONEOUS_AU_COUNT] =
      g_param_spec_uint (PROP_ERRONEOUS_AU_COUNT_ID, PROP_ERRONEOUS_AU_COUNT_ID,
          "Number of audio frames which have been marked as erroneous", 0,
          G_MAXUINT, 0, G_PARAM_READABLE | G_PARAM_STATIC_STRINGS);

  obj_properties[PROP_SUPERFRAME_PARSE_ERROR_COUNT] =
      g_param_spec_uint (PROP_SUPERFRAME_PARSE_ERROR_COUNT_ID,
          PROP_SUPERFRAME_PARSE_ERROR_COUNT_ID,
          "Number of all errors detected while disassembling the DRM super frames",
          0, G_MAXUINT, 0, G_PARAM_READABLE | G_PARAM_STATIC_STRINGS);

  obj_properties[PROP_LAST_AU_OK] =
      g_param_spec_boolean (PROP_LAST_AU_OK_ID, PROP_LAST_AU_OK_ID,
          "Flag that saves the status (ok/not ok) of the last decoded audio frame",
          FALSE, G_PARAM_READABLE | G_PARAM_STATIC_STRINGS);

  obj_properties[PROP_EXTERNAL_FRAME_COUNT] =
      g_param_spec_uint (PROP_EXTERNAL_FRAME_COUNT_ID, PROP_EXTERNAL_FRAME_COUNT_ID,
          "Frame count parameter which was attached to the buffer used to decode the current audio frame.",
          0, G_MAXUINT, 0,
          G_PARAM_READABLE | G_PARAM_STATIC_STRINGS);

  obj_properties[PROP_EXTERNAL_FRAME_CORRUPT] =
      g_param_spec_boolean (PROP_EXTERNAL_FRAME_CORRUPT_ID, PROP_EXTERNAL_FRAME_CORRUPT_ID,
          "Frame corrupt parameter which was attached to the buffer used to decode the current audio frame.",
          FALSE,
          G_PARAM_READABLE | G_PARAM_STATIC_STRINGS);

  g_object_class_install_properties (gobject_class, N_PROPERTIES, obj_properties);
}

/**
 * Initialize instance (Constructor)
 */
static void
drm_dec_init (GstDrmDec * dec, GstDrmDecClass * g_class)
{
  GST_DEBUG_OBJECT (dec, "drm_dec_init");

  /* Setup sink pad */
  GstPadTemplate *sinkPadTemplate = gst_element_class_get_pad_template (
      GST_ELEMENT_CLASS (g_class), "sink");
  if (!sinkPadTemplate) {
    GST_ELEMENT_ERROR (dec, LIBRARY, INIT, ("Failed to get sink pad template"),
        (NULL));
    return;
  }

  dec->sinkpad = gst_pad_new_from_template (sinkPadTemplate, "sink");
  gst_pad_set_chain_function (dec->sinkpad, GST_DEBUG_FUNCPTR (drm_dec_chain));
  gst_pad_set_event_function (dec->sinkpad, GST_DEBUG_FUNCPTR (drm_dec_event));
  gst_element_add_pad (GST_ELEMENT (dec), dec->sinkpad);
  GST_DEBUG_OBJECT (dec, "sinkpad created");

  /* Setup source pad */
  GstPadTemplate *sourcePadTemplate = gst_element_class_get_pad_template (
      GST_ELEMENT_CLASS (g_class), "src");
  if (!sourcePadTemplate) {
    GST_ELEMENT_ERROR (dec, LIBRARY, INIT, ("Failed to get src pad template"),
        (NULL));
    return;
  }

  dec->srcpad = gst_pad_new_from_template (sourcePadTemplate, "src");
  gst_pad_use_fixed_caps (dec->srcpad);
  gst_element_add_pad (GST_ELEMENT (dec), dec->srcpad);
  GST_DEBUG_OBJECT (dec, "srcpad created");

  /* init state */
  dec->inputQueue = g_queue_new ();
  drm_dec_reset (dec);
}

/**
 * Destructor
 */
static void
drm_dec_finalize (GObject * object)
{
  GST_INFO_OBJECT (object, "drm_dec_finalize");

  GstDrmDec * dec = GST_DRM_DEC (object);
  if (dec->localInBuffer != NULL) {
    GST_WARNING_OBJECT (object,
        "object has not been stopped before destruction");
  }

  if (dec->inputQueue) {
    g_queue_free (dec->inputQueue);
    dec->inputQueue = NULL;
  }

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

/**
 * Helper function
 * Sets the instance to a clean state for restarting processing
 */
static void
drm_dec_reset (GstDrmDec * dec)
{
  dec->bytesValid = 0;
  dec->historyLastFrameOk = FALSE;
  dec->extFrameCount = 0;
  dec->historySamples = 0;

  memset (&dec->decInfo, 0, sizeof(dec->decInfo));
  memset (&dec->audioConfig, 0, sizeof(dec->audioConfig));

  g_queue_foreach (dec->inputQueue, (GFunc) gst_buffer_unref, 0);
  g_queue_clear (dec->inputQueue);
}

/**
 * Call back function when properties are changed
 */
static void
drm_dec_set_property (GObject * object, guint prop_id, const GValue * value,
    GParamSpec * pspec)
{
  GstDrmDec *dec = GST_DRM_DEC (object);
  if (!dec)
    return;

  switch (prop_id) {
    case PROP_CONCEAL_FADEOUT_SLOPE:
      dec->decoderConfig.concealFadeoutSlope = g_value_get_uint (value);
      GST_INFO_OBJECT (dec, "Fadeout Slope property changed %i",
          dec->decoderConfig.concealFadeoutSlope);
      if (dec->drmDecoder) {
        DRMDEC_FHG_SetParam (dec->drmDecoder, DRM_AAC_CONCEAL_FADEOUT_SLOPE,
            dec->decoderConfig.concealFadeoutSlope);
      }
      break;

    case PROP_CONCEAL_FADEIN_SLOPE:
      dec->decoderConfig.concealFadeinSlope = g_value_get_uint (value);
      GST_INFO_OBJECT (dec, "Fadein Slope property changed: %i",
          dec->decoderConfig.concealFadeinSlope);
      if (dec->drmDecoder) {
        DRMDEC_FHG_SetParam (dec->drmDecoder, DRM_AAC_CONCEAL_FADEIN_SLOPE,
            dec->decoderConfig.concealFadeinSlope);
      }
      break;

    case PROP_CONCEAL_MUTE_RELEASE:
      dec->decoderConfig.concealFadeMuteRelease = g_value_get_uint (value);
      GST_INFO_OBJECT (dec, "Mute Release property changed: %i",
          dec->decoderConfig.concealFadeMuteRelease);
      if (dec->drmDecoder) {
        DRMDEC_FHG_SetParam (dec->drmDecoder, DRM_AAC_CONCEAL_MUTE_RELEASE,
            dec->decoderConfig.concealFadeMuteRelease);
      }
      break;

    case PROP_CONCEAL_COMFORT_NOISE_LEVEL:
      dec->decoderConfig.concealComfortNoiseLevel = g_value_get_uint (value);
      GST_INFO_OBJECT (dec, "Comfort Noise Level property changed: %i",
          dec->decoderConfig.concealComfortNoiseLevel);
      if (dec->drmDecoder) {
        DRMDEC_FHG_SetParam (dec->drmDecoder, DRM_AAC_CONCEAL_COMFORT_NOISE_LEVEL,
            dec->decoderConfig.concealComfortNoiseLevel);
      }
      break;

    case PROP_ACTIVATE_ULP:
      dec->decoderConfig.activateUlp = g_value_get_boolean (value);
      GST_INFO_OBJECT (dec, "ULP activation changed %i",
          dec->decoderConfig.activateUlp);
      if (dec->drmDecoder) {
        DRMDEC_FHG_SetParam (dec->drmDecoder, DRM_USAC_LP,
            dec->decoderConfig.activateUlp);
      }
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

/**
 * Callback function when querying properties from the outside
 */
static void
drm_dec_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstDrmDec *dec = GST_DRM_DEC (object);
  if (!dec)
    return;

  switch (prop_id) {
    case PROP_CONCEAL_FADEOUT_SLOPE:
      g_value_set_uint (value, dec->decoderConfig.concealFadeoutSlope);
      break;

    case PROP_CONCEAL_FADEIN_SLOPE:
      g_value_set_uint (value, dec->decoderConfig.concealFadeinSlope);
      break;

    case PROP_CONCEAL_MUTE_RELEASE:
      g_value_set_uint (value, dec->decoderConfig.concealFadeMuteRelease);
      break;

    case PROP_CONCEAL_COMFORT_NOISE_LEVEL:
      g_value_set_uint (value, dec->decoderConfig.concealComfortNoiseLevel);
      break;

    case PROP_ACTIVATE_ULP:
      g_value_set_boolean (value, dec->decoderConfig.activateUlp);
      break;

    case PROP_CURRENT_AU_SIZE:
      g_value_set_uint (value, dec->decInfo.frameSize);
      break;

    case PROP_CURRENT_BITRATE:
      g_value_set_uint (value, dec->decInfo.bitRate);
      break;

    case PROP_SUPERFRAME_COUNT:
      g_value_set_uint (value, dec->decInfo.nrSuperFrames);
      break;

    case PROP_ERRORFREE_AU_COUNT:
      g_value_set_uint (value, dec->decInfo.nrErrorFreeAUs);
      break;

    case PROP_ERRONEOUS_AU_COUNT:
      g_value_set_uint (value, dec->decInfo.nrErroneousAUs);
      break;

    case PROP_SUPERFRAME_PARSE_ERROR_COUNT:
      g_value_set_uint (value, dec->decInfo.sfParseErrors);
      break;

    case PROP_LAST_AU_OK:
       g_value_set_boolean (value, dec->decInfo.lastFrameOk);
       break;

    case PROP_EXTERNAL_FRAME_COUNT:
      g_value_set_uint (value, dec->extFrameCount);
      break;

    case PROP_EXTERNAL_FRAME_CORRUPT:
      g_value_set_boolean (value, dec->extFrameCorrupt);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static GstStateChangeReturn
drm_dec_change_state (GstElement * element, GstStateChange transition)
{
  GstDrmDec *dec = GST_DRM_DEC (element);
  if (!dec)
    return GST_STATE_CHANGE_FAILURE;

  switch (transition) {
    case GST_STATE_CHANGE_NULL_TO_READY:
      GST_TRACE_OBJECT (dec, "GST_STATE_CHANGE_NULL_TO_READY");
      break;
    case GST_STATE_CHANGE_READY_TO_PAUSED:
      GST_TRACE_OBJECT (dec, "GST_STATE_CHANGE_READY_TO_PAUSED");
      if (!drm_dec_start (dec))
        return GST_STATE_CHANGE_FAILURE;
      break;
    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
      GST_TRACE_OBJECT (dec, "GST_STATE_CHANGE_PAUSED_TO_PLAYING");
      break;
    default:
      break;
  }
  GstStateChangeReturn result = parent_class->change_state (element,
      transition);

  switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      GST_TRACE_OBJECT (dec, "GST_STATE_CHANGE_PLAYING_TO_PAUSED");
      break;
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      GST_TRACE_OBJECT (dec, "GST_STATE_CHANGE_PAUSED_TO_READY");
      if (!drm_dec_stop (dec))
        return GST_STATE_CHANGE_FAILURE;
      break;
    case GST_STATE_CHANGE_READY_TO_NULL:
      GST_TRACE_OBJECT (dec, "GST_STATE_CHANGE_READY_TO_NULL");
      break;
    default:
      break;
  }

  return result;
}

/**
 * Prepare and open resources necessary for immediate processing
 */
static gboolean
drm_dec_start (GstDrmDec * dec)
{
  coresup_init ();
  CDKprolog ();

  /* get library infos */
  LIB_INFO libInfo[CDK_MODULE_LAST];
  CDKinitLibInfo (libInfo);
  DRMDEC_FHG_GetLibInfo (libInfo);

  int i;
  for (i = 0; i < CDK_MODULE_LAST; i++) {
    if (libInfo[i].module_id != CDK_NONE) {
      GST_DEBUG_OBJECT (dec, "%s v%s (built %s, %s)",
          libInfo[i].title ? libInfo[i].title : "no title",
          libInfo[i].versionStr ? libInfo[i].versionStr : "?.?",
          libInfo[i].build_date ? libInfo[i].build_date : "no build date",
          libInfo[i].build_time ? libInfo[i].build_time : "no build time");
    }
  }

  drm_dec_get_buffer_sizes (dec, &dec->inBufferSize, &dec->outBufferSize);
  dec->inBufferSize = (guint) (dec->inBufferSize * 1.5);
  GST_DEBUG_OBJECT (dec, "Creating buffers with sizes: in %d, out %d",
      dec->inBufferSize, dec->outBufferSize);

  dec->localInBuffer = calloc (1, dec->inBufferSize);
  dec->localOutBuffer = calloc (1, dec->outBufferSize);
  if (dec->localInBuffer == NULL || dec->localOutBuffer == NULL) {
    GST_ELEMENT_ERROR (dec, LIBRARY, INIT,
        ("Open decoder - Failed to allocate buffers"), (NULL));
    return FALSE;
  }


  DRM_DECODER_ERROR result = DRMDEC_FHG_Open (&dec->drmDecoder);
  if (result != DRMDEC_OK) {
    GST_ELEMENT_ERROR (dec, LIBRARY, INIT,
        ("Open decoder - Failed with error code 0x%x", result), (NULL));
    return FALSE;
  }
  GST_INFO_OBJECT (dec, "Open decoder - Success");
  return TRUE;
}

/**
 * Read expected input and output buffer sizes from Fraunhofer library
 */
static void
drm_dec_get_buffer_sizes (GstDrmDec * dec, guint *inbuffersize,
    guint *outbuffersize)
{
  /* default Sizes */
  *inbuffersize = IN_BUF_SIZE;
  *outbuffersize = PCM_BUF_SIZE;

  CDK_bufDescr ioBufDescriptor;
  UINT bufSizes[DRMDEC_MAX_NUM_IO_BUFFERS];
  UINT eleSizes[DRMDEC_MAX_NUM_IO_BUFFERS];
  UINT bufTypes[DRMDEC_MAX_NUM_IO_BUFFERS];
  /* Initialize I/O buffer descriptor */
  ioBufDescriptor.pBufSize = bufSizes;
  ioBufDescriptor.pBufType = bufTypes;
  ioBufDescriptor.pEleSize = eleSizes;
  ioBufDescriptor.numBufs = DRMDEC_MAX_NUM_IO_BUFFERS;

  DRM_DECODER_ERROR result = DRMDEC_FHG_GetIoBufInfo (&ioBufDescriptor);
  if (result != DRMDEC_OK) {
    GST_WARNING_OBJECT (dec,
        "DRMDEC_FHG_GetIoBufInfo failed - Using default buffer sizes");
    return;
  }

  UINT bufIdx;

  /* Check input buffer */
  for (bufIdx = 0; bufIdx < ioBufDescriptor.numBufs; ++bufIdx) {
    if ((ioBufDescriptor.pBufType[bufIdx] & CDK_BUF_TYPE_INPUT)
        && (ioBufDescriptor.pBufType[bufIdx] & CDK_BUF_TYPE_BS_DATA)) {
      *inbuffersize = ioBufDescriptor.pBufSize[bufIdx]
          * ioBufDescriptor.pEleSize[bufIdx];
      break;
    }
  }
  if (bufIdx >= DRMDEC_MAX_NUM_IO_BUFFERS) {
    GST_WARNING_OBJECT (dec,
        "Input Buffer Info not found - Using default buffer size");
  }

  /* PRQA: Lint Message 648: deactivation because CDK_BUF_TYPE_OUTPUT is defined in 3rd Party API*/
  /*lint -e648 */

  /* Check output buffer */
  for (bufIdx = 0; bufIdx < ioBufDescriptor.numBufs; ++bufIdx) {
    if ((ioBufDescriptor.pBufType[bufIdx] & CDK_BUF_TYPE_OUTPUT)
        && (ioBufDescriptor.pBufType[bufIdx] & CDK_BUF_TYPE_PCM_DATA)) {
      *outbuffersize = ioBufDescriptor.pBufSize[bufIdx]
          * ioBufDescriptor.pEleSize[bufIdx];
      break;
    }
  }
  /*lint +e648 */

  if (bufIdx >= DRMDEC_MAX_NUM_IO_BUFFERS) {
    GST_WARNING_OBJECT (dec,
        "Output Buffer Info not found - Using default buffer size");
  }
}

/**
 * Release resources
 */
static gboolean
drm_dec_stop (GstDrmDec * dec)
{
  GST_INFO_OBJECT (dec, "stop");

  DRM_DECODER_ERROR result = DRMDEC_FHG_Close (&dec->drmDecoder);
  if (result != DRMDEC_OK) {
    GST_WARNING_OBJECT (dec, "DRMDEC_FHG_Close return error code %i", result);
  }
  dec->drmDecoder = 0;

  drm_dec_reset (dec);

  dec->inBufferSize = 0;
  free (dec->localInBuffer);
  dec->localInBuffer = 0;

  dec->outBufferSize = 0;
  free (dec->localOutBuffer);
  dec->localOutBuffer = 0;

  CDKepilog ();
  return TRUE;
}

/**
 * Callback for event / message receiving
 */
static gboolean
drm_dec_event (GstPad * pad, GstEvent * event)
{
  gboolean ret = TRUE;
  GstDrmDec *dec = GST_DRM_DEC (gst_pad_get_parent (pad));
  if (!dec)
    return FALSE;
  if (!dec->drmDecoder) {
    gst_object_unref (dec);
    return FALSE;
  }

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_NEWSEGMENT:
      GST_DEBUG_OBJECT (dec, "GST_EVENT_NEW_SEGMENT");
      ret = gst_pad_push_event (dec->srcpad, event);
      break;
    case GST_EVENT_EOS:
      GST_DEBUG_OBJECT (dec, "GST_EVENT_EOS");
      dec->isDraining = TRUE;
      //TODO: separate chaining and processing
      drm_dec_chain (dec->sinkpad, NULL);
      dec->isDraining = FALSE;
      ret = gst_pad_push_event (dec->srcpad, event);
      break;
    case GST_EVENT_FLUSH_STOP:
      GST_DEBUG_OBJECT (dec, "GST_EVENT_FLUSH_STOP");
      drm_dec_reset (dec);      // TODO reset decoder instance?
      ret = gst_pad_push_event (dec->srcpad, event);
      break;
    default:
      GST_DEBUG_OBJECT (dec, "other event %d", GST_EVENT_TYPE (event));
      ret = gst_pad_event_default (pad, event);
      break;
  }
  gst_object_unref (dec);
  return ret;
}

static guint
drm_dec_get_aac_sample_rate (guint32 sampleRateFromConfig)
{
  switch (sampleRateFromConfig) {
    case 1:
      return 12000;
    case 3:
      return 24000;
    case 5:
      return 48000;
    default:
      return 0;
  }
}

static guint
drm_dec_get_xheaac_sample_rate (guint32 sampleRateFromConfig)
{
  static const guint32 xheaac_sample_rates[] = { 9600, 12000, 16000, 19200,
      24000, 32000, 38400, 48000 };
  if (sampleRateFromConfig < G_N_ELEMENTS (xheaac_sample_rates))
    return xheaac_sample_rates[sampleRateFromConfig];

  return 0;

}

static guint
drm_dec_parse_audio_coding (GstDrmDec * dec)
{
  guint audioCoding = (dec->audioConfig.config[0] & CONFIG_AUDIO_CODING_BITMASK) >> 2;
  if (audioCoding != CONFIG_AAC_CODING && audioCoding != CONFIG_XHEAAC_CODING) {
    GST_ELEMENT_ERROR (dec, STREAM, WRONG_TYPE,
      ("Unexpected audio coding flag %i", audioCoding),
      (NULL));
  }
  else {
    GST_INFO_OBJECT (dec,
      "Detected DRM audioCoding field %d", audioCoding);
  }
  return audioCoding;
}

static gboolean
drm_dec_parse_sbr_flag (GstDrmDec * dec)
{
  if (dec->audioConfig.audioCoding != CONFIG_AAC_CODING)
    return FALSE;

  gboolean sbr = (dec->audioConfig.config[0] & CONFIG_SBR_BITMASK) >> 1;
  GST_INFO_OBJECT (dec,
      "Detected DRM sbr flag %d", sbr);
  return sbr;
}

static guint
drm_dec_parse_sample_rate (GstDrmDec * dec)
{
  guint result = 0;
  guint sampleRateFromConfig =
      (dec->audioConfig.config[1] & CONFIG_SAMPLE_RATE_BITMASK) >> 4;

  GST_INFO_OBJECT (dec,
      "Detected DRM sample rate field %d", sampleRateFromConfig);

  if (dec->audioConfig.audioCoding == CONFIG_AAC_CODING) {
    result = drm_dec_get_aac_sample_rate (sampleRateFromConfig);

  } else if (dec->audioConfig.audioCoding == CONFIG_XHEAAC_CODING) {
    result = drm_dec_get_xheaac_sample_rate (sampleRateFromConfig);
  }

  if (result == 0) {
    GST_ELEMENT_ERROR (dec, STREAM, WRONG_TYPE,
        ("Unexpected sample rate flag %i for audio coding %i", sampleRateFromConfig, dec->audioConfig.audioCoding),
        (NULL));
  }

  return result;
}

static guint
drm_dec_parse_channels (GstDrmDec * dec)
{
  gint audioModeFromConfig = (dec->audioConfig.config[0] & 1) << 1
      | (dec->audioConfig.config[1] & CONFING_AUDIO_MODE_BITMASK) >> 7;
  switch (audioModeFromConfig) {
    case 0: /* mono */
      return 1;
      break;
    case 1: /* parametric stereo */
    case 2: /* stereo */
      return 2;
      break;
    default:
      GST_WARNING_OBJECT (dec, "Unexpected audio mode %i", audioModeFromConfig);
      return 0;
  }
}

static void
trace_buffer (GstDrmDec * dec, GstBuffer * buffer)
{
  if (GST_BUFFER_SIZE (buffer) < 8) {
    GST_TRACE_OBJECT (dec, "Buffer too small to trace: %d bytes",
        GST_BUFFER_SIZE (buffer));
    return;
  }
  GST_TRACE_OBJECT (dec,
      "Buffer mem begin: %02x %02x %02x %02x %02x %02x %02x %02x ...",
      GST_BUFFER_DATA (buffer)[0], GST_BUFFER_DATA (buffer)[1],
      GST_BUFFER_DATA (buffer)[2], GST_BUFFER_DATA (buffer)[3],
      GST_BUFFER_DATA (buffer)[4], GST_BUFFER_DATA (buffer)[5],
      GST_BUFFER_DATA (buffer)[6], GST_BUFFER_DATA (buffer)[7]);

  guint offset = GST_BUFFER_SIZE (buffer);
  GST_TRACE_OBJECT (dec,
      "Buffer mem end  : %02x %02x %02x %02x %02x %02x %02x %02x ...",
      GST_BUFFER_DATA (buffer)[offset - 8],
      GST_BUFFER_DATA (buffer)[offset - 7],
      GST_BUFFER_DATA (buffer)[offset - 6],
      GST_BUFFER_DATA (buffer)[offset - 5],
      GST_BUFFER_DATA (buffer)[offset - 4],
      GST_BUFFER_DATA (buffer)[offset - 3],
      GST_BUFFER_DATA (buffer)[offset - 2],
      GST_BUFFER_DATA (buffer)[offset - 1]);
}

/**
 * Check if the buffer contains metadata changes.
 * Update internal state and return GST_FLOW_CUSTOM_SUCCESS
 * if metadata has changed.
 */
static GstFlowReturn
drm_dec_update_config (GstDrmDec * dec, GstBuffer * buffer)
{
  GstCaps *caps = gst_buffer_get_caps (buffer);

  if (!caps) {
    GST_WARNING_OBJECT (dec, "Buffer has no caps");
    return GST_FLOW_ERROR;
  }

  const GstStructure *structure = gst_caps_get_structure (caps, 0);
  if (!structure) {
    GST_ELEMENT_ERROR (dec, STREAM, WRONG_TYPE,
        ("Failed to read '%s' structure from buffer caps", DRM_AUDIO_MIME_TYPE), (NULL));
    gst_caps_unref (caps);
    return GST_FLOW_ERROR;
  }
  gst_caps_unref (caps);

  /* Get super frame config from buffer caps */
  const GValue *value = gst_structure_get_value (structure, AUDIO_CONFIG_ID);
  GstBuffer *sfConfig = gst_value_get_buffer (value);
  if (sfConfig == 0) {
    GST_ELEMENT_ERROR (dec, STREAM, WRONG_TYPE,
        ("Failed to read '%s' from buffer caps", AUDIO_CONFIG_ID), (NULL));
    return GST_FLOW_ERROR;
  }

  /* Get optional DRM+ flag from buffer caps */
  gboolean isDrmPlus = FALSE;
  gboolean result = gst_structure_get (structure, DRM_PLUS_ID, G_TYPE_BOOLEAN,
      &isDrmPlus, NULL);
  if (!result) {
    GST_DEBUG_OBJECT (dec,
        "Did not find '%s' in buffer caps - Assuming value 'FALSE'", DRM_PLUS_ID);
  }

  /* Get optional length higher protected part (UEP) flag from buffer caps */
  guint higherProtectedLength = 0;
  result = gst_structure_get (structure, HIGHER_PROTECTED_ID, G_TYPE_UINT,
      &higherProtectedLength, NULL);
  if (!result) {
    GST_DEBUG_OBJECT (dec,
        "Did not find '%s' in buffer caps - Assuming value '0'", HIGHER_PROTECTED_ID);
  }

  /* Get optional external frame count from buffer caps */
  guint extFrameCount = 0;
  result = gst_structure_get (structure, EXT_FRAME_COUNT_ID, G_TYPE_UINT,
      &extFrameCount, NULL);
  if (result) {
    dec->extFrameCount = extFrameCount;
  }

  /* Get optional external frame corrupt from buffer caps */
  dec->extFrameCorrupt = FALSE;
  gst_structure_get (structure, FRAME_CORRUPT_ID, G_TYPE_BOOLEAN, &dec->extFrameCorrupt,
        NULL);

  if (dec->extFrameCorrupt == TRUE)
    return GST_FLOW_OK; /* Do not update config if this flag is set */

  if (dec->audioConfig.superFrameBytes == GST_BUFFER_SIZE (buffer)
      && dec->audioConfig.lengthHigherProtected == higherProtectedLength
      && dec->audioConfig.configSize == GST_BUFFER_SIZE (sfConfig)
      && dec->audioConfig.drmPlus == isDrmPlus
      && memcmp (dec->audioConfig.config, GST_BUFFER_DATA (sfConfig), GST_BUFFER_SIZE (sfConfig)) == 0) {
    return GST_FLOW_OK;
  }

  GST_INFO_OBJECT (dec, "Configuration has changed!");

  result = drm_dec_config_dec (dec, GST_BUFFER_DATA (sfConfig), GST_BUFFER_SIZE (sfConfig),
      GST_BUFFER_SIZE (buffer), higherProtectedLength, isDrmPlus);
  if (result == FALSE)
  {
    return GST_FLOW_OK;
  }

  AudioConfig *audioConfig = &dec->audioConfig;

  audioConfig->superFrameBytes = GST_BUFFER_SIZE (buffer);
  audioConfig->lengthHigherProtected = higherProtectedLength;
  audioConfig->configSize = GST_BUFFER_SIZE (sfConfig);
  audioConfig->drmPlus = isDrmPlus;
  memcpy (audioConfig->config, GST_BUFFER_DATA (sfConfig), GST_BUFFER_SIZE (sfConfig));
  GST_INFO_OBJECT (dec, "Detected sf bytes %d, length higher protected %d, DMR+ %s",
      audioConfig->superFrameBytes, dec->audioConfig.lengthHigherProtected,
      audioConfig->drmPlus ? "true" : "false");
  if (audioConfig->configSize >= 3) {
	  GST_TRACE_OBJECT (dec, "Config mem: %02x %02x %02x",
          audioConfig->config[0], audioConfig->config[1], audioConfig->config[2]);
  }
  else {
      GST_TRACE_OBJECT (dec, "Config too small to dump: %d bytes: ", audioConfig->configSize);
  }
  if (audioConfig->configSize >= 8) {
    GST_TRACE_OBJECT (dec, " %02x %02x %02x %02x %02x",
        audioConfig->config[3], audioConfig->config[4], audioConfig->config[5],
        audioConfig->config[6], audioConfig->config[7]);
  }

  audioConfig->audioCoding = drm_dec_parse_audio_coding (dec);
  audioConfig->sbr = drm_dec_parse_sbr_flag (dec);
  audioConfig->sampleRate = drm_dec_parse_sample_rate (dec);

  GST_INFO_OBJECT (dec, "Detected sample rate %i", audioConfig->sampleRate);
  if (audioConfig->audioCoding == CONFIG_AAC_CODING && audioConfig->sbr)
  {
    audioConfig->sampleRate *= 2;
    GST_INFO_OBJECT (dec, "Detected AAC+SBR: output sample rate %i", audioConfig->sampleRate);
  }

  audioConfig->channels = drm_dec_parse_channels (dec);
  if (audioConfig->channels == 0)
    return GST_FLOW_UNEXPECTED;
  GST_INFO_OBJECT (dec, "Detected Channels %i", audioConfig->channels);

  return GST_FLOW_CUSTOM_SUCCESS;
}

/**
 * Configure Fraunhofer decoder instance according to internal state
 */
static gboolean
drm_dec_config_dec (GstDrmDec * dec, guchar* config, guint configSize, guint superFrameBytes,
    guint lengthHigherProtected, gboolean drmPlus)
{
  DRM_DECODER_ERROR result = DRMDEC_FHG_Config (dec->drmDecoder,
      config, configSize, superFrameBytes, lengthHigherProtected,
      (drmPlus ? DRM_CONFIG_DRM_PLUS : DRM_CONFIG_NONE) | DRM_CONFIG_FULL_RESET);

  if (result != DRMDEC_OK) {
    GST_WARNING_OBJECT (dec, "Config decoder - Failed % i", result);
    return FALSE;
  }

  /* DRM_CONFIG_FULL_RESET clears parameters or the property could have been set
   * before dec->drmDecoder was instantiated -> (re)apply them here */
  DRMDEC_FHG_SetParam (dec->drmDecoder, DRM_AAC_CONCEAL_FADEOUT_SLOPE,
      dec->decoderConfig.concealFadeoutSlope);
  DRMDEC_FHG_SetParam (dec->drmDecoder, DRM_AAC_CONCEAL_FADEIN_SLOPE,
      dec->decoderConfig.concealFadeinSlope);
  DRMDEC_FHG_SetParam (dec->drmDecoder, DRM_AAC_CONCEAL_MUTE_RELEASE,
      dec->decoderConfig.concealFadeMuteRelease);
  DRMDEC_FHG_SetParam (dec->drmDecoder, DRM_AAC_CONCEAL_COMFORT_NOISE_LEVEL,
      dec->decoderConfig.concealComfortNoiseLevel);
  DRMDEC_FHG_SetParam (dec->drmDecoder, DRM_USAC_LP,
      dec->decoderConfig.activateUlp);

  GST_INFO_OBJECT (dec, "Config decoder - Success");

  return TRUE;
}

static void
drm_dec_update_src_caps (GstDrmDec *dec, guint sampleRate, guint channels)
{
  GstCaps *caps = gst_caps_copy (gst_pad_get_pad_template_caps (dec->srcpad));
  gst_caps_set_simple (caps, "rate", G_TYPE_INT, sampleRate, "channels",
      G_TYPE_INT, channels, "width", G_TYPE_INT, 16, "depth", G_TYPE_INT, 16,
      "signed", G_TYPE_BOOLEAN, TRUE, NULL);

  gst_pad_set_caps (dec->srcpad, caps);
  gst_caps_unref (caps);
}

/*
 * Call this function after the incoming DRM configuration has changed.
 * It updates the source caps and resets some members
 */
static void
drm_dec_handle_changed_config (GstDrmDec * dec)
{
  drm_dec_update_src_caps (dec, dec->audioConfig.sampleRate,
      dec->audioConfig.channels);
  dec->decInfo.frameSize = 0;
  /* clean buffer which is used by Fraunhofer for concealment - avoid glitches */
  memset (dec->localOutBuffer, 0, dec->outBufferSize);
}

/**
 * Main processing function
 */
static GstFlowReturn
drm_dec_chain (GstPad * pad, GstBuffer * buffer)
{
  GstDrmDec *dec = GST_DRM_DEC (GST_PAD_PARENT (pad));
  if (!dec)
    return GST_FLOW_ERROR;

  if (buffer != NULL) {
    g_queue_push_tail (dec->inputQueue, buffer);
    guint bufferSize = GST_BUFFER_SIZE (buffer);
    guchar *inBuffer = GST_BUFFER_DATA (buffer);
    if (dec->bytesValid + bufferSize > dec->inBufferSize) {
      GST_WARNING_OBJECT (dec, "Input Buffer too small");
      return GST_FLOW_ERROR;
    }
    if (bufferSize <= 0 || inBuffer == NULL) {
      GST_WARNING_OBJECT (dec, "Input Buffer does not contain valid data");
    } else {
      memcpy (dec->localInBuffer + dec->bytesValid, inBuffer, bufferSize);
      dec->bytesValid += bufferSize;
      trace_buffer (dec, buffer);
    }
  }

  GstBuffer *currentBuffer = GST_BUFFER (g_queue_peek_head (dec->inputQueue));
  if (currentBuffer == NULL) {
    GST_INFO_OBJECT (dec, "No current Buffer");
    return GST_FLOW_OK;
  }

  if (GST_FLOW_CUSTOM_SUCCESS == drm_dec_update_config (dec, currentBuffer)) {
    drm_dec_handle_changed_config (dec);
  }

  drm_dec_handle_superframe (dec, currentBuffer);
  return GST_FLOW_OK;
}

static gint
calc_super_frame_output_samples (GstDrmDec * dec)
{
  gint superFrameSamples = dec->audioConfig.sampleRate / 5;  // 200 ms
  if (!dec->audioConfig.drmPlus) {
    superFrameSamples *= 2;     // 400 ms
  }
  return superFrameSamples;
}

static gint
calc_needed_output_audio_samples (GstDrmDec * dec)
{
  gint result = calc_super_frame_output_samples (dec) + dec->historySamples;
  GST_LOG ("Output samples needed: %d, Frames: %d", result,
      dec->decInfo.frameSize > 0 ? result / dec->decInfo.frameSize : -1);
  return result;
}

/**
 * Decode as many audio frames as necessary for the duration of one audio super frame
 * TODO: always decode as much as available. Move peek into this routine to simplify flow of processing and caps updating.
 */
static GstFlowReturn
drm_dec_handle_superframe (GstDrmDec * dec, GstBuffer * buffer)
{
  gint audioSamplesNeeded = calc_needed_output_audio_samples (dec);

  INT_PCM* localOut = dec->localOutBuffer;
  while (audioSamplesNeeded > 0) {
    guint localBytesValid = dec->bytesValid;
    GST_LOG_OBJECT (dec, "bytesValid %d, queueLength %d", dec->bytesValid,
        g_queue_get_length (dec->inputQueue));

    /* decode a single audio frame */
    gint outNumberSamples = drm_dec_decode_audio_frame (dec, dec->localInBuffer,
        &localBytesValid, localOut);

    if (dec->decInfo.numChannels > 0 && dec->audioConfig.channels != (guint)dec->decInfo.numChannels)
    {
      dec->audioConfig.channels = (guint)dec->decInfo.numChannels;
      drm_dec_update_src_caps (dec, dec->audioConfig.sampleRate, dec->audioConfig.channels);
    }

    drm_dec_push_to_out_pad (dec, localOut, outNumberSamples);
    if (dec->audioConfig.channels > 0)
      audioSamplesNeeded -= outNumberSamples / (gint) dec->audioConfig.channels;

    if (dec->historyLastFrameOk != (guint)dec->decInfo.lastFrameOk || !dec->decInfo.lastFrameOk)
    {
      g_object_notify (G_OBJECT(dec), PROP_LAST_AU_OK_ID);
    }
    dec->historyLastFrameOk = dec->decInfo.lastFrameOk;

    /* check if current buffer is consumed */
    if (localBytesValid < dec->bytesValid) {
      gint bytesConsumed = GST_BUFFER_SIZE (buffer);
      dec->bytesValid -= bytesConsumed;
      /* release current buffer */
      gst_buffer_unref (buffer);
      g_queue_pop_head (dec->inputQueue);
      /* dec->localInBuffer is not cleared here, because it will be overwritten in case a new buffer is available */

      if (g_queue_is_empty (dec->inputQueue) == FALSE) {
        /* use next buffer */
//        if (dec->bytesValid > 0) {
//          GST_WARNING_OBJECT (dec, "unexpectedly empty!");
//          break;
//        }
        buffer = GST_BUFFER (g_queue_peek_head (dec->inputQueue));

        if (GST_FLOW_CUSTOM_SUCCESS == drm_dec_update_config (dec, buffer)) {
          drm_dec_handle_changed_config (dec);
        }
        memmove (dec->localInBuffer, dec->localInBuffer + bytesConsumed,
            dec->bytesValid);
      }
    }
    if (outNumberSamples <= 0 || dec->decInfo.frameSize <= 0) {
      GST_WARNING_OBJECT (dec, "No output, leaving loop...");
      break;
    }
  }

  dec->historySamples = audioSamplesNeeded;
  return GST_FLOW_OK;
}

/**
 * Use the Fraunhofer decoder to decode a single audio frame
 */
static int
drm_dec_decode_audio_frame (GstDrmDec * dec, guchar * inBuffer, guint* bytesValid,
    INT_PCM* outBuffer)
{
  gint outNumberSamples = 0;
  DRMDEC_INFO *pDrmDecInfo = NULL;
  INT flags = DRM_DECODE_FORCE_OUTPUT;
  if (dec->isDraining == TRUE) {
    GST_DEBUG_OBJECT (dec, "Draining");
    flags = DRM_DECODE_NONE;
  }
  if (dec->extFrameCorrupt == TRUE) {
    GST_DEBUG_OBJECT (dec, "Super Frame corrupt");
    flags |= DRM_DECODE_SUPERFRAME_CORRUPT;
  }

  GST_TRACE_OBJECT (dec,
       "inBuffer begin: %02x %02x %02x %02x %02x %02x %02x %02x ...",
       inBuffer[0], inBuffer[1],
       inBuffer[2], inBuffer[3],
       inBuffer[4], inBuffer[5],
       inBuffer[6], inBuffer[7]);

  DRM_DECODER_ERROR result = DRMDEC_FHG_DecodeFrame (dec->drmDecoder, inBuffer,
      bytesValid, outBuffer, &outNumberSamples, flags, &pDrmDecInfo);

  if (result > DRMDEC_UNKNOWN) {
    GST_ELEMENT_ERROR (dec, STREAM, DECODE,
        ("DRMDEC_FHG_DecodeFrame returned error code 0x%x", result), (NULL));
  }

  GST_LOG_OBJECT (dec,
      "Decode out - inBytesValid %i, outNumberSamples %i, result %i",
      *bytesValid, outNumberSamples, result);

  GST_LOG_OBJECT (dec, "SFs %i, errFreeAUs %i, erroneousAUs %i, SFParseErrs %i, lastOK %i, bitrate %i",
      pDrmDecInfo->nrSuperFrames, pDrmDecInfo->nrErrorFreeAUs,
      pDrmDecInfo->nrErroneousAUs, pDrmDecInfo->sfParseErrors,
      pDrmDecInfo->lastFrameOk, pDrmDecInfo->bitRate);

  if (dec->decInfo.sampleRate != pDrmDecInfo->sampleRate
      || dec->decInfo.numChannels != pDrmDecInfo->numChannels
      || dec->decInfo.frameSize != pDrmDecInfo->frameSize) {
    GST_DEBUG_OBJECT (dec, "Output changed: AuSize %i, channels %i, samplerate %i",
        pDrmDecInfo->frameSize, pDrmDecInfo->numChannels,
        pDrmDecInfo->sampleRate);
  }

  dec->decInfo = *pDrmDecInfo;

  return outNumberSamples;
}

/**
 * Copy output data to new GstBuffer and push to src pad
 */
static void
drm_dec_push_to_out_pad (GstDrmDec * dec, INT_PCM* outData,
    guint outNumberSamples)
{
  guint outBufferSize = outNumberSamples * sizeof(INT_PCM);
  GstBuffer *outBuffer = NULL;
  GstFlowReturn result = gst_pad_alloc_buffer_and_set_caps (dec->srcpad,
      GST_BUFFER_OFFSET_NONE, outBufferSize, GST_PAD_CAPS (dec->srcpad),
      &outBuffer);

  if (result != GST_FLOW_OK || outBuffer == NULL) {
    GST_ELEMENT_ERROR (dec, RESOURCE, NO_SPACE_LEFT,
        ("Failed to allocate out buffer!"), (NULL));
    return;
  }
  if (GST_BUFFER_SIZE (outBuffer) < outBufferSize) {
    GST_ELEMENT_ERROR (dec, RESOURCE, FAILED, ("Wrong buffer size!"), (NULL));
    return;
  }

  memcpy (GST_BUFFER_DATA (outBuffer), outData, outBufferSize);

  gst_pad_push (dec->srcpad, outBuffer);
  return;
}

static gboolean
plugin_init (GstPlugin * plugin)
{
  GST_DEBUG_CATEGORY_INIT (drm_audio_debug, "drm-audio", GST_DEBUG_FG_CYAN, "drm-audio");

  return gst_element_register (plugin, "drm-audio", GST_RANK_PRIMARY,
      GST_TYPE_DRM_DEC);
}

GST_PLUGIN_DEFINE (
    GST_VERSION_MAJOR,
    GST_VERSION_MINOR,
    "gst_drm_audio",
    GST_DRM_DEC_DESCRIPTION,
    plugin_init,
    VERSION,
    "Proprietary",
    "ADIT",
    ""
)

/*lint +e826 +e160*/
